#!/usr/bin/env python3
"""
make_kernel_from_eigs.py

Utility to construct the base kernel required by the simulation pipeline
from the eigenvalue spectrum stored in ``kernel_eigs.csv``.  The script
reads the ``rho`` column from the CSV, tiles it as necessary to match the
expected number of lattice links (2 * L^2) and saves the resulting
one‑dimensional array to ``data/kernel.npy``.  The lattice size L is
inferred from the repository's configuration file.
"""

import os
import argparse
import yaml
import numpy as np
import pandas as pd


def determine_lattice_size(cfg: dict) -> int:
    """Extract the lattice size L from a configuration mapping.

    The configuration may define ``lattice_size`` at the top level or
    nested under a ``parameters`` section (as used in other repositories).
    """
    if 'lattice_size' in cfg:
        return int(cfg['lattice_size'])
    if 'parameters' in cfg and 'lattice_size' in cfg['parameters']:
        return int(cfg['parameters']['lattice_size'])
    raise KeyError('lattice_size not found in configuration')


def main():
    parser = argparse.ArgumentParser(description='Generate kernel.npy from kernel_eigs.csv')
    parser.add_argument('--config', default='config.yaml', help='Path to config file')
    parser.add_argument('--input_csv', default=os.path.join('data', 'kernel_eigs.csv'), help='Input CSV file')
    parser.add_argument('--output', default=os.path.join('data', 'kernel.npy'), help='Output .npy file')
    args = parser.parse_args()

    # Load configuration
    with open(args.config) as f:
        cfg = yaml.safe_load(f)
    L = determine_lattice_size(cfg)
    num_links = 2 * L * L

    # Load rho values
    df = pd.read_csv(args.input_csv)
    if 'rho' in df.columns:
        rho_vals = df['rho'].to_numpy()
    else:
        # Fallback: last numeric column
        numeric_cols = df.select_dtypes(include=['float', 'int']).columns
        rho_vals = df[numeric_cols[-1]].to_numpy()
    rho_vals = rho_vals.astype(float)

    # Tile to required length
    repeats = (num_links + len(rho_vals) - 1) // len(rho_vals)
    kernel = np.tile(rho_vals, repeats)[:num_links]

    # Save kernel
    out_path = args.output
    os.makedirs(os.path.dirname(out_path), exist_ok=True)
    np.save(out_path, kernel)
    print(f'Saved kernel to {out_path} with shape {kernel.shape} and dtype {kernel.dtype}')


if __name__ == '__main__':
    main()